#!/bin/sh

die () {
  echo "ERROR: $1" >&2
  exit 1
}

cat_repo () {
cat <<EOF
repository head.hackage
   url: http://head.hackage.haskell.org/
   secure: True
   root-keys: 07c59cb65787dedfaef5bd5f987ceb5f7e5ebf88b904bbd4c5cbdeb2ff71b740
              2e8555dde16ebd8df076f1a8ef13b8f14c66bad8eafefd7d9e37d0ed711821fb
              8f79fd2389ab2967354407ec852cbe73f2e8635793ac446d09461ffb99527f6e
   key-threshold: 3

EOF
}

# this needs to match the `remote-repo-cache` global config setting
PKGCACHE="${HOME}/.cabal/packages"

check_pkgcache () {
  if [ ! -d "$PKGCACHE" ]; then
    die "package cache folder ('$PKGCACHE') doesn't seem exist; please edit script"
  fi
}

# CABAL executable
# If $CABAL env-var is set, use that; otherwise default to the executable 'cabal'
CABAL=${CABAL:-cabal}

# GHC handling
check_ghc () {
  if [ -z "$GHC" -a -x "/opt/ghc/head/bin/ghc" ]; then
    echo "INFO: \$GHC was unset. Using the detected '/opt/ghc/head/bin/ghc' executable!"
    echo "      Set \$GHC to a different compiler if needed or edit the generated"
    echo "      'cabal.project(.local)' file accordingly"
    echo ""
    GHC="/opt/ghc/head/bin/ghc"
  fi

  # set $GHC to the name or (or full path if not in $PATH) of the GHC HEAD executable
  if [ -z "$GHC" ]; then
    die "set \$GHC with the path to your GHC HEAD executable"
  elif [ ! -f "$GHC" -a ! -x "$GHC" ]; then
    echo "WARNING: \$GHC='$GHC' does not appear to be a GHC executable"
  fi
}

############################################################################
# main

case "X$1" in
  Xupdate)
    check_pkgcache

    if "$CABAL" new-update --help > /dev/null 2>&1; then
      echo "INFO: '$CABAL' is recent enough and supports 'cabal new-update head.hackage'; you can try using that directly!"
      echo ""

      CFGFILE=$(mktemp)
      cat_repo > "$CFGFILE"

      # we need to wipe the cache for head.hackage to workaround issues in hackage-security
      rm -rf "$PKGCACHE/head.hackage/"

      echo "http-transport: plain-http" >> "$CFGFILE"

      "$CABAL" --project-file="$CFGFILE" new-update head.hackage

      rm "$CFGFILE"
    else
      echo "INFO: '$CABAL' doesn't seem to support 'cabal new-update' yet. Using legacy fallback"

      CFGFILE=$(mktemp)
      cat_repo > "$CFGFILE"

      # we need to wipe the cache for head.hackage to workaround issues in hackage-security
      rm -rf "$PKGCACHE/head.hackage/"

      echo "http-transport: plain-http" >> "$CFGFILE"
      echo "remote-repo-cache: $PKGCACHE" >> "$CFGFILE"

      "$CABAL" --config-file="$CFGFILE" update

      rm "$CFGFILE"
    fi
    ;;

  Xdump-repo)
    cat_repo
    ;;

  Xinit)
    if [ -e cabal.project ]; then
      die "cabal.project file exists already, aborting! Use '$0 init-local' to generate a 'cabal.project.local' config."
    else
      check_ghc

      echo "INFO: creating new cabal.project file"

      {
        cat <<EOF
-- generated by head.hackage.sh

with-compiler: $GHC

EOF
        echo "packages: ."
        shift
        for PKG in "$@"; do
          echo "optional-packages: $PKG"
        done

        echo ""
        cat_repo

        echo "-- end of generated content"
      } > cabal.project

    fi

    ;;


  Xinit-local)
    if [ ! -e cabal.project ]; then
      die "cabal.project does not exist, aborting!"
      exit 1
    fi

    if [ -e cabal.project.local ]; then
      die "cabal.project.local file exists already, aborting!"
    else
      check_ghc

      echo "INFO: creating new cabal.project.local file"

      {
        cat <<EOF
-- generated by head.hackage.sh

with-compiler: $GHC

EOF
        shift
        for PKG in "$@"; do
          echo "optional-packages: $PKG"
        done

        echo ""
        cat_repo

        echo "-- end of generated content"
      } > cabal.project.local

    fi

    ;;


  Xhelp)
    cat <<EOF
Usage: $0 <command>

Known <commands>

 update             update head.hackage package index

 init [folder(s)]   create cabal.project file for GHC HEAD and
                    optionally populate 'packages:' with extra folders

 init-local [folder(s)] create 'cabal.project.local' file for GHC HEAD
                    and optionally populate 'packages:' with extra folders

 dump-repo          dump 'repository' declaration to stdout and exit

 help               show help (you're here)

EOF
    ;;

  X)
    echo "no command given; try 'help'" >&2
    exit 1
    ;;

  *)
    die "unrecognised command '$1'; try 'help'"
    ;;
esac


# Local variables:
# coding: utf-8
# mode: sh
# sh-basic-offset: 2
# sh-indentation: 2
# End:
