{-# LANGUAGE OverloadedStrings #-}

module MakeConstraints where

import qualified Distribution.Package as Cabal
import Distribution.Text
import Distribution.Types.Version hiding (showVersion)
import System.Process.Typed

import qualified Data.ByteString.Lazy.Char8 as BS
import qualified Data.Set as S
import qualified Data.Map.Strict as M

import qualified Text.PrettyPrint.ANSI.Leijen as PP
import Text.PrettyPrint.ANSI.Leijen (Doc, vcat, (<+>))

import Utils

data Constraint = Installed | VersionEq Version
  deriving (Eq, Ord, Show)

-- | Each package's constraints are combined via disjunction.
newtype Constraints = Constraints (M.Map Cabal.PackageName (S.Set Constraint))

instance Semigroup Constraints where
  Constraints a <> Constraints b = Constraints $ M.unionWith (<>) a b

instance Monoid Constraints where
  mempty = Constraints mempty

constrain :: Cabal.PackageName -> Constraint -> Constraints
constrain p c = Constraints $ M.singleton p $ S.singleton c

-- These dependencies cause issues when testing boot libraries because the test-suites
-- introduce circular dependencies. One way to solve the circularity is to select
-- older version of packages (namely unix) which doesn't have the bytestring dependency (<= 2.5)
-- but we want to use the newer version of unix and just not use the optional
-- features of optparse-applicative nor tasty.
extraConstraints :: [String]
extraConstraints =
  [ "optparse-applicative -process"
  , "tasty -unix"
  ]

-- These packages we must use the installed version, because there's no way to upgrade
-- them
getBootPkgs :: FilePath
            -> IO (S.Set Cabal.PackageName)
getBootPkgs ghcPkgPath = do
  out <- readProcessStdout_ $ proc ghcPkgPath ["list", "--global", "--names-only", "--simple-output"]
  return $ S.fromList $ map Cabal.mkPackageName $ words $ BS.unpack out

constraints :: [Doc] -> Doc
constraints cs =
    "constraints:" PP.<$$> PP.indent 2 constraintsDoc
  where
    constraintsDoc = PP.vcat $ PP.punctuate "," cs

allowNewer :: S.Set Cabal.PackageName -> Doc
allowNewer pkgs =
    "allow-newer:" PP.<$$> PP.indent 2 pkgsDoc
  where
    pkgsDoc = PP.vcat $ PP.punctuate "," $ map prettyPackageName $ S.toList pkgs

renderConstraints :: Constraints -> Doc
renderConstraints (Constraints cs) =
  constraints
    [ prettyPackageName pkg <+>
        PP.hcat (PP.punctuate " || " (map renderConstraint (S.toList c)))
    | (pkg, c) <- M.toList cs
    ]

renderConstraint :: Constraint -> Doc
renderConstraint Installed = "installed"
renderConstraint (VersionEq v) = "==" <> prettyVersion v

installedConstraints :: S.Set Cabal.PackageName -> Constraints
installedConstraints bootPkgs = mconcat
  [ constrain p Installed
  | p <- S.toList bootPkgs
  ]

versionConstraints :: [(Cabal.PackageName, Version)] -> Constraints
versionConstraints pkgs = mconcat
  [ constrain p (VersionEq v)
  | (p, v) <- pkgs
  ]

makeConstraints :: FilePath -- ^ patch directory
                -> FilePath -- ^ @ghc-pkg@ path
                -> IO Doc
makeConstraints patchDir ghcPkgPath = do
  bootPkgs <- getBootPkgs ghcPkgPath
  putStrLn $ "Boot packages: " <> show bootPkgs
  patches <- findPatchedPackages patchDir
  let patchedPkgs = S.fromList $ map fst patches
      constrs = installedConstraints bootPkgs <> versionConstraints patches
      doc = PP.vcat
        [ allowNewer bootPkgs
        , ""
        , renderConstraints constrs
        , ""
        , constraints $ map PP.text extraConstraints
        ]
  return doc
